require "ev/ruby"
require "ev/ftools"
require "ev/fm"

class EVDB
  attr_reader	:tables

  def initialize(cl, ext)
    @dbclass		= cl
    @dbext		= ext
    @constraints	= nil
    @tables		= {}
    @timestamp		= nil

    if File.file?("constraints.tsv")
      @constraints	= self["constraints"]

      constraints
    end
  end

  def save
    @tables.each do |k, v|
      v.save
    end
  end

  def [](table, key=nil, fields=nil, exact=true)
    orgkey	= key

    file	= table + "." + @dbext

    if not @tables.include?(table)
      @tables[table]	= @dbclass.new(file)
    end

    table	= @tables[table]

    if table.nil?
      res	= nil
    else
      if key.nil? and fields.nil?
        res	= table
      else
        key	= key.to_s			if key.kind_of?(Numeric)
        key	= [key]				if key.kind_of?(String)
        key	= table.hash_to_row(key)	if key.kind_of?(Hash)
        key	= []				if key.nil?

        if fields.nil?
          res	= table.subset(table.headers[0..key.length-1], key, nil, exact).values.collect{|row| table.row_to_hash(row)}
        else
          res	= table.subset(table.headers[0..key.length-1], key, fields, exact).values
          res	= res.collect{|a| a[0]}	if (not res.nil? and fields.kind_of?(String))
        end

        res	= res[0]		if (not res.nil? and (orgkey.kind_of?(Numeric) or orgkey.kind_of?(String) or orgkey.kind_of?(Array)))
      end
    end

    res
  end

  def constraints
    ok	= true

	# Enhance constraints.

    @constraints.subset("Constraint", "min", ["Table", "Column", "Constraint", "Value"]).each_value do |ta, co, ch, va|
      a	= [ta, co, "num", va]
      @constraints[a]	= a
    end

    @constraints.subset("Constraint", "max", ["Table", "Column", "Constraint", "Value"]).each_value do |ta, co, ch, va|
      a	= [ta, co, "num", va]
      @constraints[a]	= a
    end

    if ok
      @constraints.subset(nil, nil, ["Table", "Column", "Constraint", "Value"]).each_value do |ta, co, ch, va|
        if not ["key", "num", "min", "max", "mut"].include?(ch)
          melding(ta, co, ch, va, "Unkonwn check \"%s\"." % [ch])
	  ok	= false
        end
      end
    end

    if ok
      @constraints.subset(nil, nil, ["Table", "Column", "Constraint", "Value"]).each_value do |ta, co, ch, va|
        if not File.file?(ta + "." + @dbext)
          melding(ta, co, ch, va, "Table \"%s\" doesn't exist." % [ta])
          ok	= false
        end
      end
    end

    if ok
      @constraints.subset(nil, nil, ["Table", "Column", "Constraint", "Value"]).each_value do |ta, co, ch, va|
        if not self[ta].headers.include?(co)
          melding(ta, co, ch, va, "Table \"%s\" has no column \"%s\"." % [ta, co])
	  ok	= false
        end
      end
    end

    if ok
      @constraints.subset("Constraint", "key", ["Table", "Column", "Constraint", "Value"]).each_value do |ta, co, ch, va|
        if not File.file?(va + "." + @dbext)
          melding(ta, co, ch, va, "Table \"%s\" doesn't exist." % [va])
          ok	= false
        end
      end
    end

    if ok
      @constraints.subset("Constraint", "key", ["Table", "Column", "Constraint", "Value"]).each_value do |ta, co, ch, va|
        self[ta].subset(nil, nil, co).each_value do |a|
          if not a[0].empty? and not self[va].has_key?(a)
            melding(ta, co, ch, va, "Table \"%s\" has no \"%s\"." % [va, a])
	    ok	= false
          end
        end
      end
    end

    if ok
      @constraints.subset("Constraint", "num", ["Table", "Column", "Constraint", "Value"]).each_value do |ta, co, ch, va|
        self[ta].subset(nil, nil, co).each_value do |a|
          if not a[0].numeric?
            melding(ta, co, ch, va, "%s \"%s\" is not numeric." % [co, a[0]])
	    ok	= false
          end
        end
      end
    end

    if ok
      @constraints.subset("Constraint", "min", ["Table", "Column", "Constraint", "Value"]).each_value do |ta, co, ch, va|
        self[ta].subset(nil, nil, co).each_value do |a|
          if not a[0].to_f >= va.to_f
            melding(ta, co, ch, va, "%s is less than %s." % [a[0], va])
	    ok	= false
          end
        end
      end
    end

    if ok
      @constraints.subset("Constraint", "max", ["Table", "Column", "Constraint", "Value"]).each_value do |ta, co, ch, va|
        self[ta].subset(nil, nil, co).each_value do |a|
          if not a[0].to_f >= va.to_f
            melding(ta, co, ch, va, "%s is more than %s." % [a[0], va])
	    ok	= false
          end
        end
      end
    end

    if ok
      @constraints.subset("Constraint", "mut", ["Table", "Column", "Constraint", "Value"]).each_value do |ta, co, ch, va|
        if not File.file?(va + "." + @dbext)
          melding(ta, co, ch, va, "Table \"%s\" doesn't exist." % [va])
          ok	= false
        end
      end
    end

    if ok
      @constraints.subset("Constraint", "mut", ["Table", "Column", "Constraint", "Value"]).each_value do |ta, co, ch, va|
        h1	= self[ta].headers.dup
        h2	= self[va].headers.dup
        h1.delete(co)
        if h1 != h2
          melding(ta, co, ch, va, "Headers don't match.")
	  ok	= false
        end
      end
    end
  end

  def melding(ta, co, ch, va, s)
    puts "%-40s [\"%s\", \"%s\", \"%s\", \"%s\"]" % [s, ta, co, ch, va]
  end

  def mut(timestamp=nil)
    @timestamp	= timestamp

    unless @constraints.nil?
      @constraints.subset("Constraint", "mut", ["Table", "Column", "Value"]).each_value do |ta, co, va|
        @tables.delete(va)

        if @timestamp.nil?
          @tables[va]	= self[va].mut(self[ta])
        else
          @tables[va]	= self[va].deep_dup
          0.upto(@timestamp) do |n|
            @tables[va]	= self[va].mut(self[ta].subset(co, n.to_s))
          end
        end
      end
    end

    self
  end
end

class EVTable < Hash
  attr_reader :headers
  attr_writer :headers
  attr_reader :comments
  attr_writer :comments
  attr_reader :key
  attr_reader :data
  attr_writer :data
  attr_reader :sep
  attr_writer :sep
  attr_reader :file
  attr_writer :file

  def initialize(file=nil)
    @file	= file
    @headers	= []
    @comments	= []
    @key	= 0
    @data	= 0
    @sep	= "\t"
  end

  def key=(k)
    refresh(k)
  end

  def refresh(k=@key)
    h	= to_hash
    @key	= k
    from_hash(h)
  end

  def to_hash
    res	= self.dup.clear
    each do |k, v|
      res[key_to_hash(k)]	= row_to_hash(v)
    end
    res
  end

  def from_hash(h, default=nil)
    self.clear

    h.each do |k, v|
      k2	= []
      v2	= []

      @headers.length.times do |n|
        s	= v[@headers[n]]
        s	= default	if s.nil?

        k2	<< s	if n < @key
        v2	<< s
      end

      self[k2]	= v2
    end
  end

  def hash_to_key(hash)
    @headers[0..@key-1].collect{|k| hash[k]}
  end

  def key_to_hash(k)
    row_to_hash(k[0..@key-1])
  end

  def hash_to_row(hash)
    @headers.collect{|k| hash[k]}
  end

  def row_to_hash(r)
    res	= {}
    row	= self[r[0..@key-1]]

    if row.nil?
      res	= nil
    else
      0.upto(@headers.length-1) do |n|
        res[@headers[n]]	= row[n]
      end
    end

    res
  end

  def save(file=@file)
    @file	= file

    res		= {}
    each do |k, v|
      k.collect!{|s| s.gsub(/\n/ , "\\n")}
      k.collect!{|s| s.gsub(/\\n/, "\\n")}
      v.collect!{|s| s.gsub(/\n/ , "\\n")}
      v.collect!{|s| s.gsub(/\\n/, "\\n")}
      res[k]	= v
    end
    self.replace(res)
  end

  def add(k, value)
    k.collect!		{|s| s.nil? ? "" : s}
    value.collect!	{|s| s.nil? ? "" : s}

    k	= value	if k.empty?

    self[k]	= value
  end

  def find(string)
    res		= nil
    if string.nil? or string.empty?
      res	= self.dup
    else
      res	= self.dup.delete_if{|k, v| not v.join("\0").downcase.include?(string.downcase)}
    end
    res
  end

  def fuzzy(string, drempel)
    res		= nil
    if string.nil? or string.empty?
      res	= self.dup
    else
      res	= self.dup.delete_if{|k, v| not (v.collect{|s| s.downcase.fm(string.downcase) > drempel}.include?(true))}
    end
    res
  end

  def subset(fields, values, results=nil, exact=true, emptyline=nil, joinwith=nil)
    fields	= fields.dup	unless fields.nil?
    values	= values.dup	unless values.nil?
    results	= results.dup	unless results.nil?

    fields	= [fields]	if not fields.kind_of? Array
    values	= [values]	if not values.kind_of? Array
    results	= [results]	if not results.kind_of? Array

    res	= {}
    res	= self.clone
    res.clear
    key	= values[0...@key]

    fields.each_index do |c|
      fields[c]	= @headers[fields[c]]	unless (fields[c].nil? or fields[c].kind_of?(String))
    end

    if fields == @headers and self.include?(key)
      a	= self[key]
      b	= []

      results.each_index do |r|
        r		= @headers.index(results[r])	if results[r].kind_of?(String)
        b << a[r]
      end

      res[key]	= b
    else
      fields.each_index do |c|
        fields[c]	= @headers.index(fields[c])	if fields[c].kind_of?(String)
      end

      results.each_index do |r|
        results[r]	= @headers.index(results[r])	if results[r].kind_of?(String)
      end

      res	= super
    end

    return res
  end

  def mut(muttable)
    res	= self.deep_dup

    muttable.subset(nil, nil, @headers).each do |k, v|
      k	= v[0..@key-1]

      v.length.times do |n|
        res[k][n]	= v[n]	if ((not v[n].nil?) and (not v[n].empty?))
      end
    end

    res
  end

  def insert_column(place, name, default="", key=@key)
    res	= nil

    if (not @headers.include?(name)) and place <= @headers.length+1 and key <= @headers.length+1
      h	= self.to_hash

      @headers[place, 0]	= name
      @key			= key

      self.from_hash(h, default)

      res	= self
    end

    res
  end

  def delete_column(name, key=@key)
    res	= nil

    if @headers.include?(name) and key <= @headers.length-1
      h	= self.to_hash

      @headers.delete(name)
      @key	= key

      self.from_hash(h)

      res	= self
    end

    res
  end

  def keyheaders
    @headers[0..(@key-1)]
  end

  def inspect
    "(%s, %s)" % [headers.inspect, super]
  end

  def to_tsv(file)
    db	= TSVFile.new

    db.file	= file
    db.headers	= @headers
    db.comments	= []
    db.key	= @key
    db.sep	= @sep

    @headers.each_index do |h|
      db.comments << "# #{h+1} #{@headers[h]}"
    end
    db.comments << "# SEP #{@sep}"

    db.replace(self)

    db
  end

  def to_html(args=nil, highlight=[])
    highlight	= [highlight]	unless highlight.kind_of?(Array)

    highlight.collect! do |a|
      a.kind_of?(Array) ? a : [a]
    end

    res	= ""

    unless args.nil?
      res << "<table %s>" % args
    else
      res << "<table>"
    end

    res << "<tr>"

    @headers.each do |header|
      res << "<th>%s</th>" % header
    end

    res << "</tr>\n"

    sort.each do |k, v|
      if highlight.include?(k)
        res << "<tr bgcolor='silver'>"
      else
        res << "<tr>"
      end

      teller	= 0
      v.each do |field|
        teller += 1

        field	= "&nbsp;"	if field.empty?

        if teller <= @key
          res << "<td><b>%s</b></td>" % field
        else
          res << "<td>%s</td>" % field
        end
      end

      res << "</tr>\n"
    end

    res << "</table>\n"

    res
  end
end

class TSVFile < EVTable
  def initialize(file=nil)
    super

    if not file.nil?
      File.rollbackup(file).readlines.each do |line|
        line.chomp!

        if line =~ /^\#/
          fields	= line.split(" ")

          if fields.length > 2 and fields[1].numeric?
            @headers[fields[1].to_i-1]	= fields[2]
            @data += 1
          else
            @comments.push line
          end

          @comments.delete(line)	if ["KEY", "SEP"].include?(fields[1])

          if fields.length == 3
            @key	= fields[2].to_i	if fields[1] == "KEY"
            @sep	= fields[2]		if fields[1] == "SEP"
          end
        else
          fields	= line.split(@sep, -1).collect{|s| s.gsub(/\\n/, "\n")}

          if @key.zero?
            k	= fields
          else
            k	= fields[0..(@key-1)]
          end

          self[k]	= (fields + ([""] * (@headers.length-1)))[0..@headers.length-1]
        end
      end
    end
  end

  def save(file=@file)
    super

    File.rollbackup(file, "w") do |f|
      @headers.length.times do |n|
        f.puts "# %s %s" % [n+1, @headers[n]]
      end

      f.puts "# KEY %s" % @key
      f.puts "# SEP %s" % @sep

      @comments.each		{|s| f.puts s}
      self.keys.sort.each	{|s| f.puts self[s].join(@sep)}
    end
  end
end

class TabFile < EVTable
  def initialize(file=nil)
    super

    if not file.nil?
      header	= true

      File.rollbackup(file).readlines.each do |line|
        line.chomp!

        if line =~ /^\#/
          @comments.push line
        else
          fields	= line.split(@sep, -1).collect{|s| s.gsub(/\\n/, "\n")}

          if header
            @headers		= fields
            @data		= fields.length
            header		= false
          else
            self[fields]	= fields
          end
        end
      end
    end
  end

  def save(file=@file)
    super

    File.rollbackup(file, "w") do |f|
      f.puts @headers.join(@sep)

      self.keys.sort.each	{|s| f.puts self[s].join(@sep)}
    end
  end
end

class BlockFile < EVTable
  def initialize(file=nil)
    super

    if not file.nil?
      lines	= File.rollbackup(file).readlines.chomp.join("\n")
      blocks	= lines.split(/\n\n/)

      blocks.shift.split("\n").each do |line|
        l		= line.split(" ")
        @headers[l[0].to_i-1]	= l[1]
      end

      blocks.each do |block|
        buffer	= Array.new
        nr	= nil

        block.split("\n").each do |line|
          if line.numeric?
            nr	= line.to_i
          else
            buffer[nr-1]	= Array.new	if buffer[nr-1].nil?
            buffer[nr-1]	<< line
          end
        end

        if buffer.include?(nil)
          puts block
          p buffer
        end

        buffer[2]	= [buffer[2].join(" ")]	if file =~ /recepten\.txt/	# TODO

        fields		= buffer.collect{|b| b.join("\n")}
        self[fields]	= fields
      end
    end
  end
end
